/**
 * Copyright Notice
 *
 * This is a work of the U.S. Government and is not subject to copyright
 * protection in the United States. Foreign copyrights may apply.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package gov.vha.isaac.utils.file_transfer;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.util.concurrent.Callable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gov.vha.isaac.utils.file_transfer.FileTransfer.FileActionResult;
import net.lingala.zip4j.core.ZipFile;
import net.lingala.zip4j.progress.ProgressMonitor;

/**
 * 
 * {@link Download}
 *
 * @author <a href="mailto:joel.kniaz.list@gmail.com">Joel Kniaz</a>
 *
 */
public class Download implements Callable<File>
{
	private static Logger log = LoggerFactory.getLogger(Download.class);
	String username_, password_;
	URL url_;
	private String altFileBaseName_;
	private boolean cancel_ = false;
	private boolean unzip_;
	private boolean ignoreChecksumFail_;
	private boolean overwrite_;
	private File targetFolder_;

	/**
	 * @param username (optional) used if provided
	 * @param password (optional) used if provided
	 * @param url The URL to download from
	 * @param altFileBaseName Alternate basename of the downloaded file
	 * @param unzip - Treat the file as a zip file, and unzip it after the download
	 * @param ignoreChecksumFail - If a checksum file is found on the repository - don't fail if the downloaded file doesn't match the expected value.
	 * (If no checksum file is found on the repository, this option is ignored and the download succeeds)
	 * @param overwrite boolean indicating whether existing files should be overwritten
	 * @param targetFolder (optional) download and/or extract into this folder.  If not provided, a folder 
	 * will be created in the system temp folder for this purpose.
	 * @throws IOException 
	 */
	public Download(
			String username,
			String password,
			URL url,
			String altFileBaseName,
			boolean unzip,
			boolean ignoreChecksumFail,
			boolean overwrite,
			File targetFolder) throws IOException
	{
		this.username_ = username;
		this.password_ = password;
		this.url_ = url;
		this.altFileBaseName_ = altFileBaseName;
		this.unzip_ = unzip;
		this.targetFolder_ = targetFolder;
		this.ignoreChecksumFail_ = ignoreChecksumFail;
		this.overwrite_ = overwrite;
		if (targetFolder_ == null)
		{
			targetFolder_ = File.createTempFile("ISAAC", "");
			targetFolder_.delete();
		}
		else
		{
			targetFolder_ = targetFolder_.getAbsoluteFile();
		}
		targetFolder_.mkdirs();
	}

	/* (non-Javadoc)
	 * @see java.util.concurrent.Callable#call()
	 */
	public File call() throws Exception
	{
		File fileFromUrl = getFileFromUrl(url_);
		
		// If file already successfully handled THIS SESSION then return file
		if (FileTransfer.hasBeenSuccessfullyHandled(fileFromUrl.getAbsoluteFile().toString())) {
			log.debug("Not downloading already downloaded file from " + url_);

			return fileFromUrl;
		}

		// Attempt to download file from url
		File downloadedFile = download(url_);

		String calculatedSha1Value = null;
		String expectedSha1Value = null;
		
		// If not a file ending with ".sha1" or ".md5" then download SHA1 file and validate checksum
		if (! url_.toString().endsWith(".sha1") && ! url_.toString().endsWith(".md5")) {
			try
			{
				// Download MD5 file
				URL sha1FileUrl = new URL(url_.toString() + ".sha1");
				if (FileTransferUtils.remoteFileExists(sha1FileUrl, username_, password_)) {
					// Download SHA1 file
					File sha1File = download(sha1FileUrl);
					// Read SHA1 value from SHA1 file
					expectedSha1Value = Files.readAllLines(sha1File.toPath()).get(0);
					// Calculate SHA1 value of target file
					calculatedSha1Value = ChecksumGenerator.calculateChecksum("SHA1", downloadedFile);
				}
			}
			catch (Exception e1)
			{
				log.debug("Failed to get .sha1 file for " + url_.toString(), e1);
			}
			if (calculatedSha1Value != null && !expectedSha1Value.equals(calculatedSha1Value))
			{
				// If ! ignoreChecksumFail_ is true then fail by throwing exception
				if (! ignoreChecksumFail_) 
				{
					throw new RuntimeException("Checksum of downloaded file '" + url_.toString() + "' does not match the expected value!");
				}
				else
				{
					log.warn("Checksum of downloaded file '" + url_.toString() + "' does not match the expected value!");
				}
			}
			try
			{
				// Download MD5 file
				URL md5Url = new URL(url_.toString() + ".md5");
				if (FileTransferUtils.remoteFileExists(md5Url, username_, password_)) {
					download(md5Url);
				}
			}
			catch (Exception e1)
			{
				log.warn("Failed to get .md5 file for " + url_.toString(), e1);
			}
			// If altFileBaseName_ is not null then rename downloaded file to altFileBaseName_
			if (altFileBaseName_ != null && ! altFileBaseName_.equals(downloadedFile.getName())) {
				File newFile = new File(downloadedFile.getParentFile().getAbsoluteFile(), FileTransferUtils.basename(altFileBaseName_));
				try {
					Files.move(downloadedFile.toPath(), newFile.toPath(), StandardCopyOption.ATOMIC_MOVE, StandardCopyOption.REPLACE_EXISTING);
					downloadedFile = newFile;
				} catch (Exception e) {
					log.error("Failed changing " + downloadedFile.getName() + " to " + newFile, e);
					throw new RuntimeException("Failed changing " + downloadedFile.getName() + " to " + newFile + ". Caught " + e.getClass().getName() + " " + e.getLocalizedMessage());
				}
			}
		}
		
		if (cancel_)
		{
			log.debug("Download cancelled of " + url_.toString());
			throw new Exception("Cancelled download of " + url_);
		}
		
		// If unzip_ true then unzip downloaded file
		if (unzip_)
		{
			try
			{
				ZipFile zipFile = new ZipFile(downloadedFile);
				zipFile.setRunInThread(true);
				zipFile.extractAll(targetFolder_.getAbsolutePath());
				while (zipFile.getProgressMonitor().getState() == ProgressMonitor.STATE_BUSY)
				{
					if (cancel_)
					{
						zipFile.getProgressMonitor().cancelAllTasks();
						log.debug("Download cancelled");
						throw new Exception("Cancelled!");
					}
					try
					{
						//TODO see if there is an API where I don't have to poll for completion
						Thread.sleep(25);
					}
					catch (InterruptedException e)
					{
						// noop
					}
				}
				log.debug("Unzip complete");
			}
			catch (Exception e)
			{
				log.error("error unzipping", e);
				throw new Exception("The downloaded file doesn't appear to be a zip file");
			}
			finally
			{
				downloadedFile.delete();
			}
			return targetFolder_;
		}
		else
		{
			return downloadedFile;
		}
	}
	
	private File getFileFromUrl(URL url) {
		String fileName = url.toString();
		fileName = fileName.substring(fileName.lastIndexOf('/') + 1, fileName.length());
		File file = new File(targetFolder_, fileName);
		
		return file;
	}
	
	private File download(URL url) throws Exception
	{
		HttpURLConnection httpCon = null;
		
		// Construct local target file from url, but do not open or download
		File file = getFileFromUrl(url);
		
		if (FileTransfer.hasBeenSuccessfullyHandled(file.getAbsoluteFile().toString())) {
			// If file already downloaded THIS SESSION then return it
			log.debug("Not downloading already downloaded (this session) file from " + url);

			return file;
		} else {
			if (! file.getName().endsWith(".sha1") && ! file.getName().endsWith(".md5")) {
				// If file suffix is not ".sha1" or ".md5" then guarantee that the SHA1 file is downloaded
				// but do not download again unless overwrite_ is true
				if (FileTransferUtils.downloadChecksumFilesAndValidateChecksum(file, url, username_, password_) && ! overwrite_) {
					log.debug("Not downloading already downloaded file (previous session) from " + url);

					// Register file as successfully handled
					FileTransfer.setFileActionResult(file.getAbsolutePath(), FileActionResult.EXISTS);
					return file;
				}
			}
			
			// Register file as unsuccessfully handled. Change to success only after completion and validation
			FileTransfer.setFileActionResult(file.getAbsolutePath(), FileActionResult.FAILED);
		}

		log.debug("Beginning download from " + url);
		try {
			httpCon = NetworkUtils.getConnection(url, username_, password_);
			httpCon.setDoInput(true);
			httpCon.setRequestMethod("GET");
			httpCon.setConnectTimeout(30 * 1000);
			httpCon.setReadTimeout(60 * 60 * 1000);
			InputStream in = httpCon.getInputStream();

			int bytesReadSinceShowingProgress = 0;
			byte[] buf = new byte[1048576];
			FileOutputStream fos= new FileOutputStream(file);
			try {
				int read = 0;
				while ((read = in.read(buf, 0, buf.length)) > 0)
				{
					if (cancel_ || Thread.interrupted()) {
						throw new InterruptedException();
					}
					fos.write(buf, 0, read);

					bytesReadSinceShowingProgress += buf.length;
					if (bytesReadSinceShowingProgress >= FileTransfer.IO_SHOW_PROGRESS_BYTES_THRESHOLD) {
						ConsoleUtil.showProgress(); // Show progress for buffered I/O
						bytesReadSinceShowingProgress = 0;
					}

					if (FileTransfer.DEBUG_FAIL_PROCESSING_FILE_NAME != null && file.getName().equals(FileTransfer.DEBUG_FAIL_PROCESSING_FILE_NAME)) {
						throw new IOException("intentionally failing download of \"" + file.getName() + "\" for debugging purposes");
					}
				}
				fos.flush();
			} catch (Exception e) {
				// Register file as unsuccessfully handled
				FileTransfer.setFileActionResult(file.getAbsoluteFile().toString(), FileActionResult.FAILED);

				if (e instanceof InterruptedException) {
					log.error("Failed writing to file " + file.getAbsolutePath());
				} else {
					log.error("Failed writing to file " + file.getAbsolutePath(), e);
				}
				file.delete();
				throw e;
			} finally {
				in.close();
				fos.close();
			}

			if (! file.getName().endsWith(".sha1") && ! file.getName().endsWith(".md5")) {
				// If file does not end with ".sha1" or ".md5" then validate against previously-downloaded checksum file
				if (! FileTransferUtils.validateChecksum(file, new File(file.getAbsoluteFile() + ".sha1"))) {
					if (! ignoreChecksumFail_) {
						// Fail if checksum verification fails and ignoreChecksumFail_ is false
						throw new Exception("Verify checksum FAILED of " + file.getAbsolutePath());
					}
				}
			}
				
			if (cancel_)
			{
				log.debug("Download cancelled of " + file.getAbsoluteFile());
				throw new Exception("Cancelled download of " + file.getAbsoluteFile());
			}
			else
			{
				// Register file as successfully handled
				FileTransfer.setFileActionResult(file.getAbsoluteFile().toString(), FileActionResult.SUCCEEDED);

				log.debug("Download complete of " + file.getAbsoluteFile());
			}
			return file;
		} finally {
			if (httpCon != null) {
				httpCon.disconnect();
			}
		}
	}
}